/*
	Copyright (C) 2004 Christopher E. Miller
	
	This software is provided 'as-is', without any express or implied
	warranty.  In no event will the authors be held liable for any damages
	arising from the use of this software.
	
	Permission is granted to anyone to use this software for any purpose,
	including commercial applications, and to alter it and redistribute it
	freely, subject to the following restrictions:
	
	1. The origin of this software must not be misrepresented; you must not
	   claim that you wrote the original software. If you use this software
	   in a product, an acknowledgment in the product documentation would be
	   appreciated but is not required.
	2. Altered source versions must be plainly marked as such, and must not be
	   misrepresented as being the original software.
	3. This notice may not be removed or altered from any source distribution.
*/


private import std.string, std.path;
private import std.c.stdio;
private import std.c.windows.windows;

private import cderr;


extern(Windows)
{
	DWORD CommDlgExtendedError();
}


enum FilePromptFlags: DWORD
{
	ALLOWMULTISELECT = 0x00000200,
	CREATEPROMPT = 0x00002000,
	EXTENSIONDIFFERENT = 0x00000400,
	FILEMUSTEXIST = 0x00001000,
	HIDEREADONLY = 0x00000004,
	NOCHANGEDIR = 0x00000008,
	NODEREFERENCELINKS = 0x00100000,
	NONETWORKBUTTON = 0x00020000,
	NOREADONLYRETURN = 0x00008000,
	NOTESTFILECREATE = 0x00010000,
	OVERWRITEPROMPT = 0x00000002,
	PATHMUSTEXIST = 0x00000800,
	READONLY = 0x00000001
}


class FilePromptException: OpenFileNameDlgErr
{
	this(char[] msg)
	{
		super(msg);
	}
	
	
	this(uint errorCode)
	{
		super(errorCode);
	}
}


class FilePrompt
{
	OPENFILENAMEA ofn;
	uint fileLen = 0;
	
	
	this(HINSTANCE hinstance)
	{
		ofn.hInstance = hinstance;
		ofn.lStructSize = ofn.sizeof;
		ofn.Flags = 0x00080000; // OFN_EXPLORER
	}
	
	
	this()
	{
		this(GetModuleHandleA(null));
	}
	
	
	void flags(FilePromptFlags flags)
	{
		ofn.Flags = cast(DWORD)flags | 0x00080000; // (flags | OFN_EXPLORER)
	}
	
	
	bit open(HWND hwndOwner, char[] filter, char[] customFilter, uint filterIndex, char[] file, char[] initialDir, char[] title, char[] defExt)
	{
		//char* p;
		
		ofn.hwndOwner = hwndOwner;
		//p = &filter[0] + filter.length;
		//if(!p[0] && !p[1])
		//{
		//	ofn.lpstrFilter = filter;
		//}
		//else
		{
			ofn.lpstrFilter = new char[filter.length + 2];
			ofn.lpstrFilter[0 .. filter.length] = filter;
			ofn.lpstrFilter[filter.length .. filter.length + 2] = "\0\0";
		}
		if(customFilter.length)
		{
			ofn.nMaxCustFilter = customFilter.length + 128;
			ofn.lpstrCustomFilter = new char[ofn.nMaxCustFilter];
			ofn.lpstrCustomFilter[0 .. customFilter.length] = customFilter;
			ofn.lpstrCustomFilter[customFilter.length] = 0;
		}
		else
		{
			ofn.nMaxCustFilter = 0;
			ofn.lpstrCustomFilter = null;
		}
		ofn.nFilterIndex = filterIndex;
		ofn.nMaxFile = file.length + 2048;
		ofn.lpstrFile = new char[ofn.nMaxFile];
		ofn.lpstrFile[0 .. file.length] = file;
		ofn.lpstrFile[file.length] = 0;
		ofn.nMaxFileTitle = 0;
		ofn.lpstrFileTitle = null;
		ofn.lpstrInitialDir = !initialDir.length ? null : toStringz(initialDir);
		ofn.lpstrTitle = !title.length ? null : toStringz(title);
		ofn.lpstrDefExt = !defExt.length ? null : toStringz(defExt);
		
		if(!GetOpenFileNameA(&ofn))
		{
			DWORD error = CommDlgExtendedError();
			if(!error)
				return false;
			throw new FilePromptException(error);
		}
		fileLen = strlen(ofn.lpstrFile);
		return true;
	}
	
	
	bit open(HWND hwndOwner, char[] filter, char[] file) //shortcut
	{
		return open(hwndOwner, filter, null, 1, file, null, null, null);
	}
	
	
	bit save(HWND hwndOwner, char[] filter, char[] customFilter, uint filterIndex, char[] file, char[] initialDir, char[] title, char[] defExt)
	{
		//char* p;
		
		ofn.hwndOwner = hwndOwner;
		//p = &filter[0] + filter.length;
		//if(!p[0] && !p[1])
		//{
		//	ofn.lpstrFilter = filter;
		//}
		//else
		{
			ofn.lpstrFilter = new char[filter.length + 2];
			ofn.lpstrFilter[0 .. filter.length] = filter;
			ofn.lpstrFilter[filter.length .. filter.length + 2] = "\0\0";
		}
		if(customFilter.length)
		{
			ofn.nMaxCustFilter = customFilter.length + 128;
			ofn.lpstrCustomFilter = new char[ofn.nMaxCustFilter];
			ofn.lpstrCustomFilter[0 .. customFilter.length] = customFilter;
			ofn.lpstrCustomFilter[customFilter.length] = 0;
		}
		else
		{
			ofn.nMaxCustFilter = 0;
			ofn.lpstrCustomFilter = null;
		}
		ofn.nFilterIndex = filterIndex;
		ofn.nMaxFile = file.length + 2048;
		ofn.lpstrFile = new char[ofn.nMaxFile];
		ofn.lpstrFile[0 .. file.length] = file;
		ofn.lpstrFile[file.length] = 0;
		ofn.nMaxFileTitle = 0;
		ofn.lpstrFileTitle = null;
		ofn.lpstrInitialDir = !initialDir.length ? null : toStringz(initialDir);
		ofn.lpstrTitle = !title.length ? null : toStringz(title);
		ofn.lpstrDefExt = !defExt.length ? null : toStringz(defExt);
		
		if(!GetSaveFileNameA(&ofn))
		{
			DWORD error = CommDlgExtendedError();
			if(!error)
				return false;
			throw new FilePromptException(error);
		}
		fileLen = strlen(ofn.lpstrFile);
		return true;
	}
	
	
	bit save(HWND hwndOwner, char[] filter, char[] file) //shortcut
	{
		return save(hwndOwner, filter, null, 1, file, null, null, null);
	}
	
	
	char[] file() //may be one or more with open(), depending on flags and user selection
	{
		return ofn.lpstrFile[0 .. fileLen];
	}
	
	
	char[] fileTitle() //shortcut, only useful for one file
	{
		return getBaseName(file());
	}
}


char[][] splitFiles(char[] files)
{
	char[][] result;
	uint i;
	bit inQuote = false;
	bit findStart = true;
	uint startIndex = 0;
	
	for(i = 0; i != files.length; i++)
	{
		if(findStart)
		{
			if(files[i] == ' ' || files[i] == '\t')
				continue;
			findStart = false;
			startIndex = i;
		}
		
		if(files[i] == '"')
		{
			inQuote = !inQuote;
			if(!inQuote) //matched quotes
			{
				result.length = result.length + 1;
				result[result.length - 1] = files[startIndex .. i];
				findStart = true;
			}
			else //starting quote
			{
				if(startIndex != i) //must be a quote stuck to another word, separate them
				{
					result.length = result.length + 1;
					result[result.length - 1] = files[startIndex .. i];
					startIndex = i + 1;
				}
				else
				{
					startIndex++; //exclude the quote
				}
			}
		}
		else if(!inQuote)
		{
			if(files[i] == ' ' || files[i] == '\t')
			{
				result.length = result.length + 1;
				result[result.length - 1] = files[startIndex .. i];
				findStart = true;
			}
		}
	}
	
	if(startIndex != i)
	{
		result.length = result.length + 1;
		result[result.length - 1] = files[startIndex .. i];
	}
	
	return result;
}


unittest
{
	char[] files = `foo bar "hello world"! arf"?" :)`;
	char[][] sf = splitFiles(files);
	
	assert(sf[0] == "foo");
	assert(sf[1] == "bar");
	assert(sf[2] == "hello world");
	assert(sf[3] == "!");
	assert(sf[4] == "arf");
	assert(sf[5] == "?");
	assert(sf[6] == ":)");
	assert(sf.length == 7);
}

