/*
	DFL HTMLget written by Christopher E. Miller
	This code is public domain.
	You may use it for any purpose.
	This code has no warranties and is provided 'as-is'.
*/

// To compile:
// 	dfl dflhtmlget -gui

// or:
// 	set dfl_flags=-version=DFL_TangoNetDeviceBerkeley
// 	dfl dflhtmlget -gui -dfl-build -version=DFL_TangoNetDeviceBerkeley

// or:
// 	set dfl_flags=-version=Old
// 	dfl dflhtmlget -gui -dfl-build -version=Old
// Note: Tango itself will need to be rebuilt with -version=Old to use Old.


version(Old)
{
	import tango.net.Socket;
}
else
{
	import tango.net.device.Berkeley;
}
import tango.net.Uri;

import tango.text.Util;
import tango.text.Ascii;
import tango.text.convert.Integer;

import dfl.all;


class MainForm: Form
{
	RichTextBox disp; // HTML display.
	TextBox urlBox;
	Button okb;
	
	char[] url;
	char[] domain;
	uint addr;
	ushort port;
	
	
	this()
	{
		text = "DFL HTMLget";
		
		Label label;
		with(label = new Label)
		{
			label.text = "&Address:";
			bounds = Rect(4, 3, 50, 14);
			parent = this;
		}
		
		with(urlBox = new TextBox)
		{
			text = "http://www.digitalmars.com/d/";
			maxLength = 1000;
			left = 58;
			// width set in layout event.
			parent = this;
		}
		
		with(okb = new Button)
		{
			text = "Go";
			width = 35;
			// left set in layout event.
			parent = this;
			
			click ~= &okb_click; // Register click handler.
		}
		acceptButton = okb; // Set this button as the accept (default) button.
		
		with(disp = new RichTextBox)
		{
			font = new Font("Courier New", 10f);
			disp.maxLength = int.max;
			detectUrls = false; // Note: may want to detect and handle them later.
			location = Point(0, okb.bottom);
			// size set in layout event.
			parent = this;
		}
		
		layout ~= &form_layout; // Register layout handler to move and resize controls.
		
		size = Size(600, 400);
	}
	
	
	// Layout handler.
	private void form_layout(Object sender, LayoutEventArgs ea)
	{
		urlBox.width = this.clientSize.width - urlBox.left - okb.width;
		okb.left = urlBox.right;
		disp.size = Size(this.clientSize.width, this.clientSize.height - disp.top);
	}
	
	
	// Click handler for OK button.
	private void okb_click(Object sender, EventArgs ea)
	{
		try
		{
			// Disable controls that may interfere.
			okb.enabled = false;
			disp.enabled = false;
			
			disp.text = null; // Empty HTML display.
			
			// Get typed URL.
			url = urlBox.text;
			url = tango.text.Util.trim(url);
			url = tango.net.Uri.Uri.encode(url, tango.net.Uri.Uri.IncGeneric);
			
			// Parse the URL...
			int i;
			
			i = tango.text.Util.locatePattern!(char)(url, "://");
			if(i < url.length)
			{
				if(tango.text.Ascii.icompare(url[0 .. i], "http"))
					throw new Exception("http:// expected");
				url = url[i + 3 .. url.length]; // Strip off protocol.
			}
			
			i = tango.text.Util.locate!(char)(url, '#');
			if(i < url.length) // Remove anchor ref.
				url = url[0 .. i];
			
			i = tango.text.Util.locate!(char)(url, '/');
			if(i >= url.length)
			{
				domain = url;
				url = "/";
			}
			else
			{
				domain = url[0 .. i];
				url = url[i .. url.length];
			}
			// -url- is now the server virtual URI.
			assert(url[0] == '/');
			
			i = tango.text.Util.locate!(char)(domain, ':');
			if(i >= domain.length)
			{
				port = 80; // Default HTTP port.
			}
			else
			{
				int iport = tango.text.convert.Integer.toInt!(char)(domain[i + 1 .. domain.length]);
				if(iport < 0 || iport > ushort.max)
				{
					throw new Exception("Port out of range");
				}
				port = cast(ushort)iport;
				domain = domain[0 .. i];
			}
			
			disp.selectedText = "Connecting to " ~ domain ~ " on port " ~ tango.text.convert.Integer.toString(port) ~ "...\r\n";
			
			// See if it's an IP address.
			addr = IPv4Address.parse(domain);
			if(IPv4Address.ADDR_NONE == addr)
			{
				// It is not an IP address, so resolve as hostname.
				// Resolve it asynchronously so that we don't block the GUI...
				// Choose what to use as a callback delegate.
				asyncGetHostByName(domain, &gettingHostCallback);
			}
			else
			{
				// It is an IP address, so connect to it!
				doConnect();
			}
		}
		catch(DflThrowable e)
		{
			disp.selectedText = "Error: " ~ e.toString() ~ "\r\n";
			reEnable();
		}
	}
	
	
	void reEnable()
	{
		// Re-enable disabled controls.
		okb.enabled = true;
		disp.selectionStart = 0;
		disp.enabled = true;
	}
	
	
	void gettingHostCallback(NetHost inetHost, int err)
	{
		if(err)
		{
			// Oops, failed to lookup the hostname.
			disp.selectedText = "Unable to resolve " ~ domain ~ "\r\n";
			
			// Go back and let the user try another.
			reEnable();
		}
		else
		{
			// Resolved, connect to it!
			addr = inetHost.addrList[0];
			doConnect();
		}
	}
	
	
	void doConnect()
	{
		// Reset state variable.
		gotHeader = false;
		
		// Setup an acynchronous socket.
		AsyncTcpSocket sock;
		sock = new AsyncTcpSocket;
		
		// Setup a read/write queue for the socket.
		// -queue- is declared as private below.
		queue = new SocketQueue(sock);
		
		// Choose which events I want and what to use as a callback delegate.
		sock.event(EventType.CONNECT | EventType.READ | EventType.WRITE | EventType.CLOSE, &onSocketEvent);
		
		// Finally, initiate connection!
		sock.connect(new IPv4Address(addr, port));
	}
	
	
	void onSocketEvent(DflSocket sock, EventType type, int err)
	{
		if(!queue)
			return; // In case the connection closed while a socket event was waiting.
		
		switch(type)
		{
			case EventType.READ: // Time to receive.
				// Let the queue read the data first.
				queue.readEvent();
				
				// Check the queue.
				onRead();
				if(!queue)
					return;
				
				// Make sure there's not too much data in the queue.
				if(queue.receiveBytes() > 0x4000)
				{
					// If too much data received, abort the connection.
					// Don't want to consume too much memory.
					onClose();
					
					// Report what happened...
					msgBox(this, "Too much data");
				}
				break;
			
			case EventType.WRITE: // Able to send more data; this event is not re-sent unless you send something.
				// Let the queue handle writing data I've added to it.
				queue.writeEvent();
				break;
			
			case EventType.CONNECT:
				onConnect();
				break;
			
			case EventType.CLOSE:
				// Can be more to read upon remote disconnection.
				queue.readEvent();
				onRead();
				if(!queue)
					return;
				
				onClose();
				
				// Report what happened if connection closed too early.
				if(!gotHeader)
					msgBox(this, "Connection closed");
				break;
			
			default: ;
		}
	}
	
	
	void onConnect()
	{
		// Note: may want to add a timer to make sure we get data in a timely manner.
		
		// Queue up the HTTP request.
		sendRequest();
	}
	
	
	void sendRequest()
	{
		// Send the HTTP request via the queue.
		
		disp.selectedText = "Connected!\r\nRequesting URL \"" ~ url ~ "\"...\r\n";
		
		char[] request;
		
		request = "GET " ~ url ~ " HTTP/1.0\r\n"
			"Accept-Charset: utf-8\r\n"
			"Host: " ~ domain;
		if(port != 80)
			request ~= ":" ~ tango.text.convert.Integer.toString(port);
		request ~= "\r\n\r\n";
		
		//msgBox(request);
		
		// Finally, add it to the queue to be sent!
		queue.send(request);
	}
	
	
	// Note: if non UTF-8 is received, it will result in an exception;
	// better handling should be added, but this is just an example.
	
	void onRead()
	{
		// Time to check the queue.
		
		int i;
		char[] s;
		
		again:
		if(!gotHeader)
		{
			// Not removing from the queue if there's not an entire line yet.
			// Just peek at it for now.
			s = cast(char[])queue.peek();
			
			const char[] HTTP_EOL = "\r\n";
			
			i = tango.text.Util.locatePattern!(char)(s, HTTP_EOL);
			if(i < s.length)
			{
				// Found HTTP_EOL.
				char[] line;
				line = s[0 .. i];
				
				// Remove the line from the queue first.
				queue.receive(i + HTTP_EOL.length); // Removes -i- bytes plus the HTTP_EOL.
				
				if(!line.length)
				{
					// If the line is empty, it's the end of the header.
					gotHeader = true;
					
					// Reset display to prepare for HTML.
					disp.text = null;
				}
				
				// Process the line.
				const char[] CONTENT_TYPE_NAME = "Content-Type: ";
				if(line.length > CONTENT_TYPE_NAME.length &&
					!tango.text.Ascii.icompare(CONTENT_TYPE_NAME, line[0 .. CONTENT_TYPE_NAME.length]))
				{
					char[] type;
					type = line[CONTENT_TYPE_NAME.length .. line.length];
					if(type.length <= 5 || tango.text.Ascii.icompare("text/", type[0 .. 5]))
					{
						// URL is not text.
						// Close the connection early.
						onClose();
						
						// Report what happened...
						msgBox(this, "URL is not text");
						
						return;
					}
				}
				
				// Read some more.
				//if(queue.peek().length)
					goto again;
			}
		}
		else // gotHeader == true
		{
			// Not removing from the queue if there's not an entire tag yet.
			// Just peek at it for now.
			s = cast(char[])queue.peek();
			
			/+
			disp.selectedText = s;
			queue.receive();
			return;
			+/
			
			//if(s.length)
			{
				size_t startiw = size_t.max, iw;
				new_tag:
				for(iw = 0; iw != s.length; iw++)
				{
					switch(s[iw])
					{
						case '<':
							if(startiw == size_t.max)
								startiw = iw;
							break;
						
						case '>':
							if(startiw != size_t.max)
							{
								if(iw - startiw >= 4 && // See if it's a comment.
									s[startiw + 1] == '!' && s[startiw + 2] == '-' && s[startiw + 3] == '-')
								{
									// It's a comment..
									if(s[iw - 1] == '-' && s[iw - 2] == '-')
									{
										if(startiw)
											disp.selectedText = s[0 .. startiw];
										
										disp.selectionColor = Color(0x7F, 0x7F, 0x7F);
										disp.selectedText = s[startiw .. iw + 1];
									}
									else
									{
										// Need to keep looking for the end.
										continue;
									}
								}
								else
								{
									if(startiw)
										disp.selectedText = s[0 .. startiw];
									
									disp.selectionColor = Color(0xBB, 0, 0);
									disp.selectedText = s[startiw .. iw + 1];
								}
								disp.selectionColor = disp.foreColor;
								
								// Start looking for another tag.
								startiw = size_t.max;
								queue.receive(iw + 1); // Remove these bytes from the queue.
								s = s[iw + 1 .. s.length];
								goto new_tag;
							}
							break;
						
						default: ;
					}
				}
			}
		}
	}
	
	
	void onClose()
	{
		if(gotHeader)
		{
			// Need to empty the queue of received data in case onRead()
			// left some there due to an incomplete tag.
			if(queue.receiveBytes())
				disp.selectedText = cast(char[])queue.receive();
		}
		
		// Re-enable disabled controls.
		reEnable();
		
		// Clean up.
		queue.socket.detach();
		queue = null;
	}
	
	
	private:
	SocketQueue queue; // Only valid while connected.
	bool gotHeader;
}


int main()
{
	int result = 0;
	
	try
	{
		Application.enableVisualStyles();
		
		Application.run(new MainForm);
	}
	catch(DflThrowable o)
	{
		msgBox(o.toString(), "Fatal Error", MsgBoxButtons.OK, MsgBoxIcon.ERROR);
		
		result = 1;
	}
	
	return result;
}

